/* $Id: autorange.c,v 1.27 1999/07/02 15:54:29 dhiller Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Thoroughly test auto-range function */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"
#include "err1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE_MAX	16384

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: autorange.c,v 1.27 1999/07/02 15:54:29 dhiller Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list, int *vibrato)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    /* Do we have E1432 or E1433? */
    *vibrato = 0;
    for (i = 0; i < nmod; i++)
	if (hwconfig[i].model_code == E1432_MODEL_CODE_E1432)
	    *vibrato = 1;

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      double clock_freq, double *span, int data_mode, long blocksize,
      int fft, int mode)
{
    FLOATSIZ32 tmpspan;

    CHECK(e1432_set_blocksize(hw, group, blocksize));
    CHECK(e1432_set_data_mode(hw, group, data_mode));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    if (fft)
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));
    CHECK(e1432_set_clock_freq(hw, group, clock_freq));
    CHECK(e1432_set_auto_range_mode(hw, group, mode));

    CHECK(e1432_set_span(hw, group, *span));
    CHECK(e1432_get_span(hw, group, &tmpspan));
    *span = tmpspan;

    return 0;
}

static int
start_meas(E1432ID hw, int group)
{
    CHECK(e1432_init_measure(hw, group));
    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, double clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5, rv;

    rv = 0;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	rv = -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	rv = -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	rv = -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	rv = -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	rv = -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	rv = -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	rv = -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	rv = -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	rv = -1;
    }

    return rv;
}

static int
wait_block_avail(E1432ID hw, int group, int verbose,
		 long blocksize, double span, int long_wait)
{
    clock_t start, timeout;
    int     status;

    timeout = (2.5 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (long_wait)
	timeout *= 3;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 1)
	(void) printf("Block available found\n");

    return 0;
}

static int
get_scan(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	 double clock_freq, long blocksize, int fft, int verbose,
	 double span, int vibrato)
{
    FLOATSIZ64 buffer[BLOCKSIZE_MAX];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    int     type, chan;

    /* Wait for block available */
    /* Vibrato is really slow at this stuff when doing FFTs, because
       it takes a long time to discard all the lost scans after an
       autorange.  Allow that without error, as long as the wait
       doesn't get TOO long. */
    CHECK(wait_block_avail(hw, group, verbose, blocksize, span,
			   vibrato && fft));

    /* Read the data */
    for (type = 0; type < (fft ? 2 : 1); type++)
	for (chan = 0; chan < nchan; chan++)
	{
	    if (verbose > 3)
		(void) printf("Reading chan %d type %d\n", chan, type);

	    CHECK(e1432_read_float64_data(hw, chan_list[chan],
					  type == 0 ? E1432_TIME_DATA :
					  E1432_FREQ_DATA, buffer,
					  blocksize, &trailer,
					  &count));
	    if (count != blocksize)
	    {
		(void) fprintf(stderr,
			       "%s: e1432_read_float64_data: "
			       "actual count was %ld, expected %ld\n",
			       progname, count, blocksize);
		return -1;
	    }

	    CHECK(check_trailer(&trailer, clock_freq, span,
				chan_list[chan], type));
	}

    return 0;
}

static int
do_range(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	 double *signal_list, double *min_range, double *max_range,
	 double clock_freq, double span, double ar_time,
	 long blocksize, int fft, int verbose, int running,
	 double range, int each_chan, int vibrato)
{
    FLOATSIZ32 result;
    SHORTSIZ16 comm, diff, half;
    double  ratio;
    int     i, error, rv, status;

    if (verbose > 0)
	(void) printf("Starting range %6.3f\n", range);

    /* We might be asking for a range the hardware can't do */
    CHECK(e1432_print_errors(0));
    status = e1432_set_range(hw, group, range);
    CHECK(e1432_print_errors(1));
    if (status < 0)
	if (status != ERR1432_RANGE_OUT_OF_LIMITS)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_set_range: returned %d\n",
			   progname, status);
	    return status;
	}
	else
	{
	    if (verbose > 0)
		(void) printf("Skipping range %6.3f\n", range);
	    return 0;
	}

    if (running)
	CHECK(get_scan(hw, group, nchan, chan_list, clock_freq,
		       blocksize, fft, verbose, span, vibrato));

    if (each_chan)
	for (i = 0; i < nchan; i++)
	    CHECK(e1432_auto_range(hw, chan_list[i], ar_time));
    else
	CHECK(e1432_auto_range(hw, group, ar_time));

    rv = 0;
    for (i = 0; i < nchan; i++)
    {
	CHECK(e1432_get_range(hw, chan_list[i], &result));
	if (verbose > 1)
	    (void) printf("Chan %2d Range %6.3f -> %6.3f",
			  i, range, result);
	if (running)
	{
	    /* Only check overloads if measurement is running */
	    CHECK(e1432_check_overloads(hw, chan_list[i], NULL,
					&comm, &diff, &half));
	    if (verbose > 1)
		(void) printf(" (%s)", (comm || diff) ? "overload" :
			      half ? "fullscale" : "underrange");
	}
	if (verbose > 1)
	    (void) printf("\n");

	/* If we know what the expected range is, check that the
	   auto-range got reasonably close to it. */
	if (signal_list[i] > 0)
	{
	    if (signal_list[i] < min_range[i])
		ratio = min_range[i] / result;
	    else if (signal_list[i] > max_range[i])
		ratio = max_range[i] / result;
	    else
		ratio = signal_list[i] / result;

	    /* We should be roughly within one range step of the
	       signal.  The largest range step is between 2 and 5, a
	       factor of 2.5.  This may need to be widened for E1433,
	       which has a very high overload level. */
	    if (ratio < 0.4 || ratio > 2.5)
	    {
		(void) fprintf(stderr,
			       "%s: auto-range failed: "
			       "chan %d, start %g, expected %g, got %g\n",
			       progname, i + 1, range,
			       signal_list[i], result);
		rv = -1;
	    }

	    if (running)
	    {
		error = 0;
		if (signal_list[i] <= min_range[i] / 2)
		    /* Should be underranged */
		    if (comm || diff || half)
			error = 1;
		if (signal_list[i] > min_range[i] / 2 &&
		    signal_list[i] < max_range[i] * 2)
		    /* Should be full-scale */
		    if (comm || diff || !half)
			error = 1;
		if (signal_list[i] >= max_range[i] * 2)
		    if (!diff || !half)
			error = 1;
		if (error)
		{
		    (void) fprintf(stderr,
				   "%s: auto-range failed: "
				   "chan %d, start %g, "
				   "comm %d, diff %d, half %d\n",
				   progname, i + 1, range,
				   comm, diff, half);
		    rv = -1;
		}
	    }
	}
    }

    if (running)
	CHECK(get_scan(hw, group, nchan, chan_list,clock_freq,
		       blocksize, fft, verbose, span, vibrato));

    return rv;
}

/*ARGSUSED*/
static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    double *signal_list, double clock_freq, double span,
    double ar_time, double range, long blocksize, int fft,
    int verbose, int running, int each_chan, int vibrato)
{
    FLOATSIZ32 min, max, dummy;
    double  min_range[NCHAN_MAX], max_range[NCHAN_MAX];
    int     chan;

    for (chan = 0; chan < nchan; chan++)
	if (signal_list[chan] >= 0)
	{
	    CHECK(e1432_get_range_limits(hw, chan_list[chan],
					 &min, &max, &dummy, &dummy));
	    /* The min limit will probably be zero.  Check what it
	       gets rounded to when actually set. */
	    CHECK(e1432_set_range(hw, chan_list[chan], min));
	    CHECK(e1432_get_range(hw, chan_list[chan], &min));
	    min_range[chan] = min;
	    max_range[chan] = max;

	    if (verbose > 0)
		(void) printf("Chan %d signal level %g\n",
			      chan + 1, signal_list[chan]);
	}

    if (range < 0)
    {
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 0.005, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 0.01, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 0.02, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 0.05, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 0.1, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 0.2, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 0.5, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 1.0, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 2.0, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 5.0, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 10.0, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, 20.0, each_chan, vibrato));
    }
    else
    {
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, range, each_chan, vibrato));
	CHECK(do_range(hw, group, nchan, chan_list,
		       signal_list, min_range, max_range, clock_freq,
		       span, ar_time, blocksize, fft, verbose,
		       running, range, each_chan, vibrato));
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
	"Usage: %s [-bcCfuvV] [-L laddr] [-m mode] [-n nchan] [-N nmod]\n"
		   "\t[-r range] [-s span] [-t time] [-x lev[,lev]]\n"
		   "\t-b: Use block mode\n"
		   "\t-c: Use continuous mode\n"
		   "\t-C: Auto-range each channel separately\n"
		   "\t-f: Set up for fft data\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-m: Auto range mode 0=normal, 1=up, 2=down\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-r: Test <range>, default all ranges\n"
		   "\t-s: Use <span>, default 20000 Hz\n"
		   "\t-t: Auto-Range <time> sec per range, default auto\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n"
		   "\t-x: Specify expected signal levels\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    double  signal_list[NCHAN_MAX];
    double  ar_time, range, span;
    char   *p;
    long    blocksize;
    int     c, i, data_mode, fft, nmod, verbose, chan, offset, count;
    int     group, nchan, each_chan, mode, vibrato;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
    blocksize = 1024;
    each_chan = 0;
    fft = 0;
    laddr[0] = 8;
    mode = E1432_AUTO_RANGE_MODE_DEF;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    range = -1;			/* Meaning test all ranges */
    span = 20000;
    ar_time = 0.0;
    verbose = 0;
    for (chan = 0; chan < NCHAN_MAX; chan++)
	signal_list[chan] = -1;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "bB:cCfL:m:n:N:r:s:t:uvVx:")) != -1)
	switch (c)
	{
	case 'b':
	    data_mode = E1432_BLOCK_MODE;
	    break;
	case 'B':
	    blocksize = strtol(optarg, &p, 0);
	    if (optarg == p || blocksize <= 0 || blocksize > BLOCKSIZE_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid blocksize: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'c':
	    data_mode = E1432_CONTINUOUS_MODE;
	    break;
	case 'C':
	    each_chan = 1;
	    break;
	case 'f':
	    fft = 1;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    mode = (int) strtol(optarg, &p, 0);
	    if (optarg == p || mode < 0 || mode > 2)
	    {
		(void) fprintf(stderr,
			       "%s: invalid auto-range mode: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    switch (mode)
	    {
	    case 0:
		mode = E1432_AUTO_RANGE_MODE_DEF;
		break;
	    case 1:
		mode = E1432_AUTO_RANGE_MODE_UP;
		break;
	    case 2:
		mode = E1432_AUTO_RANGE_MODE_DOWN;
		break;
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'r':
	    if (sscanf(optarg, "%lg", &range) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid range: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 's':
	    if (sscanf(optarg, "%lg", &span) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid span: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 't':
	    if (sscanf(optarg, "%lg", &ar_time) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid auto-range time: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'x':
	    offset = 0;
	    for (chan = 0; chan < NCHAN_MAX; chan++)
	    {
		if (sscanf(optarg + offset, "%lg%n",
			   &signal_list[chan], &count) != 1)
		{
		    if (strlen(optarg + offset) > 0)
		    {
			(void) fprintf(stderr,
				       "%s: invalid expected signal level: '%s'\n",
				       progname, optarg);
			usage();
		    }
		    break;
		}
		offset += count;
		count = 0;
		(void) sscanf(optarg + offset, " ,%n", &count);
		offset += count;
	    }
	    break;
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list, &vibrato) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list,
	      51200.0, &span, data_mode, blocksize, fft, mode) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, signal_list,
	    51200.0, span, ar_time, range, blocksize, fft, verbose, 0,
	    each_chan, vibrato) < 0)
	return EXIT_FAILURE;
    if (start_meas(hw, group) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, signal_list,
	    51200.0, span, ar_time, range, blocksize, fft, verbose, 1,
	    each_chan, vibrato) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
